# Dental Dashboard - Form Submission Structure

## Overview
The dental order submission system has been reorganized with proper field mapping according to the database structure defined in `structure.txt`.

## File Organization

### `inc/dental.php`
- **Purpose**: Main dental order management and submission handling
- **Functions**:
  - `handle_dental_order_submission()` - AJAX handler for order submission
  - `process_dental_order_submission()` - Main order processing with field mapping
  - `get_user_info()` - Get user information for orders
  - `update_files_to_permanent()` - Update files from temp to permanent status
  - `get_dental_file_info()` - Get file information by ID

### `inc/file-upload-handlers.php`
- **Purpose**: File upload handling only (no longer contains order submission code)
- **Functions**:
  - `handle_dental_file_upload()` - AJAX handler for file uploads
  - `handle_dental_file_delete()` - AJAX handler for file deletion
  - `validate_uploaded_file()` - File validation
  - `process_file_upload()` - File upload processing
  - `delete_dental_file()` - File deletion
  - `cleanup_old_temp_files()` - Cleanup old temporary files

## Form Field Mapping

### Database Field Mapping
| Form Field | Database Field | Type | Description |
|------------|----------------|------|-------------|
| `service_type` | `services.service_type` | Text | Main service type |
| `sub_services[]` | `services.sub_services` | JSON Array | Selected sub-services from 2nd step |
| `removable_services[]` | `services.extras` | JSON Array | Checkbox selections (comma separated) |
| `patient_name` | `patient_name` | Text | Patient's name |
| `patient_age` | `dental_meta.patient_age` | Meta | Patient's age (custom meta) |
| `order_note` | `order_note` | Text | Order notes |
| `attachments` | `attachments` | JSON Array | Complete uploaded file information |

### Services JSON Structure
```json
{
  "service_type": "main_service_name",
  "sub_services": ["sub_service_1", "sub_service_2"],
  "extras": ["extra_service_1", "extra_service_2"]
}
```

### Author Info JSON Structure
```json
{
  "name": "First Last",
  "phone": "phone_meta",
  "address": "address_meta",
  "address_note": "address_note_meta",
  "geo_location": "geo_location_meta"
}
```

### Dental ID Format
- Format: `d-{number}` (e.g., `d-1001`, `d-1002`, etc.)
- Starts from 1001 and increments automatically
- Generated using `generate_dental_id()` function

### Secret Key Format
- 8 characters long
- Contains only lowercase letters and numbers
- Generated using `generate_secret_key()` function
- Example: `a1b2c3d4`

### Attachments JSON Structure
```json
[
  {
         "file_id": "file_abc123def456",
     "original_name": "Patient_Scan_John_Doe.jpg",
     "stored_name": "a1b2c3d4e5f6_1705123456.jpg",
     "file_path": "2025/08/john_doe/dental_d-1001",
     "file_url": "https://sv1.etiuse.com/dental-files/2025/08/john_doe/dental_d-1001/a1b2c3d4e5f6_1705123456.jpg",
    "file_size": 2048576,
    "mime_type": "image/jpeg",
    "upload_date": "2025-01-15 10:30:00",
    "uploaded_by": 123,
    "is_temp": 0,
    "hash": "a1b2c3d4e5f6"
  }
]
```

## File Management

### File Storage Structure
- All files are stored in permanent directory: `/dental-files/{year}/{month}/{username}/`
- Files uploaded during form filling are marked with `is_temp = 1`
- On form submission, files are updated to `is_temp = 0` and moved to: `/dental-files/{year}/{month}/{username}/dental_{dental_id}/`
- Files are associated with specific dental order

### File Update Process
1. User uploads files during form filling (stored in permanent directory with `is_temp = 1`)
2. On form submission, files are updated to `is_temp = 0` and path is updated to include dental_id
3. File paths and URLs are updated in database
4. Attachments array is updated with final file information

### Folder Naming Convention
- Uses actual username instead of "user_" + ID
- Fallback to "user_" + ID if username is not available
- Example: `/dental-files/2025/01/john_doe/dental_d-1001/`

## AJAX Endpoints

### File Upload
- **Action**: `upload_dental_file`
- **Method**: POST
- **Parameters**:
  - `file`: Uploaded file
  - `user_id`: Current user ID
  - `dental_id`: (optional) Dental order ID for permanent storage
  - `nonce`: Security nonce

### File Deletion
- **Action**: `delete_dental_file`
- **Method**: POST
- **Parameters**:
  - `file_id`: File ID to delete
  - `nonce`: Security nonce

### Order Submission
- **Action**: `submit_dental_order`
- **Method**: POST
- **Parameters**:
  - `service_type`: Main service type
  - `sub_services[]`: Array of sub-services
  - `removable_services[]`: Array of extra services
  - `patient_name`: Patient name
  - `patient_age`: Patient age
  - `order_note`: Order notes
  - `attachments`: JSON array of complete uploaded file information
  - `nonce`: Security nonce

## Security Features

### Nonce Verification
- All AJAX endpoints require valid nonces
- Prevents CSRF attacks

### File Validation
- File type validation (MIME type + extension)
- File size limits (200 MB max)
- Secure file naming and storage

### User Authentication
- All operations require logged-in user
- User ID validation for file operations

## Database Integration

### Main Table: `wp_dentals`
- Stores order information
- JSON fields for complex data (services, attachments, etc.)

### Meta Table: `wp_dental_meta`
- Stores additional order metadata
- Flexible key-value storage

### File Storage
- Files stored on FTP server
- File metadata stored in WordPress options (temporary solution)
- Future: Move to dedicated file management table

## Error Handling

### Validation Errors
- Field validation with specific error messages
- File validation with detailed error reporting

### Upload Errors
- FTP connection failures
- File permission issues
- Database storage failures

### Order Processing Errors
- Database insertion failures
- File movement failures
- Meta data storage failures

## Future Improvements

1. **Dedicated File Table**: Replace WordPress options with proper file management table
2. **File Compression**: Add support for file compression
3. **Batch Operations**: Support for batch file operations
4. **File Preview**: Add file preview functionality
5. **Version Control**: Add file versioning support
