<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Disable the emoji's
 */
function disable_emojis() {
    remove_action( 'wp_head', 'print_emoji_detection_script', 7 );
    remove_action( 'admin_print_scripts', 'print_emoji_detection_script' );
    remove_action( 'wp_print_styles', 'print_emoji_styles' );
    remove_action( 'admin_print_styles', 'print_emoji_styles' ); 
    remove_filter( 'the_content_feed', 'wp_staticize_emoji' );
    remove_filter( 'comment_text_rss', 'wp_staticize_emoji' ); 
    remove_filter( 'wp_mail', 'wp_staticize_emoji_for_email' );
    add_filter( 'tiny_mce_plugins', 'disable_emojis_tinymce' );
    add_filter( 'wp_resource_hints', 'disable_emojis_remove_dns_prefetch', 10, 2 );
}
add_action( 'init', 'disable_emojis' );

/**
* Filter function used to remove the tinymce emoji plugin.
* 
* @param array $plugins 
* @return array Difference betwen the two arrays
*/
function disable_emojis_tinymce( $plugins ) {
    if ( is_array( $plugins ) ) {
        return array_diff( $plugins, array( 'wpemoji' ) );
    } else {
        return array();
    }
}

/**
* Remove emoji CDN hostname from DNS prefetching hints.
*
* @param array $urls URLs to print for resource hints.
* @param string $relation_type The relation type the URLs are printed for.
* @return array Difference betwen the two arrays.
*/
function disable_emojis_remove_dns_prefetch( $urls, $relation_type ) {
    if ( 'dns-prefetch' == $relation_type ) {
    /** This filter is documented in wp-includes/formatting.php */
    $emoji_svg_url = apply_filters( 'emoji_svg_url', 'https://s.w.org/images/core/emoji/2/svg/' );

    $urls = array_diff( $urls, array( $emoji_svg_url ) );
    }

    return $urls;
}


/**
 * Disable access to the /wp-json/wp/v2/users endpoint for non-authenticated users
 */
add_filter('rest_endpoints', function ($endpoints) {
    if (isset($endpoints['/wp/v2/users'])) {
        // Remove the endpoint to restrict access
        unset($endpoints['/wp/v2/users']);
    }
    return $endpoints;
});

/**
 * Hiding WordPress Admin Bar
*/ 
add_filter('show_admin_bar', '__return_false');

/**
 * Logout without Confiramtion
*/ 
function logout_without_confirm($action, $result) {
    if ($action == "log-out" && !isset($_GET['_wpnonce'])) {
        $redirect_to = isset($_REQUEST['redirect_to']) ? $_REQUEST['redirect_to'] : home_url();
        $location = str_replace('&amp;', '&', wp_logout_url($redirect_to));
        header("Location: $location");
        die;
    }
}
add_action('check_admin_referer', 'logout_without_confirm', 10, 2);

/**
 * Disabling Author Archives
*/
function disable_author_archive() {
    if (is_author()) {
        global $wp_query;
        $wp_query->set_404();
        status_header(404);
        nocache_headers();
    }
}
add_action('template_redirect', 'disable_author_archive');

/**
 * Disable Comments
 */
add_action('admin_init', function () {
    // Redirect any user trying to access comments page
    global $pagenow;
     
    if ($pagenow === 'edit-comments.php') {
        wp_safe_redirect(admin_url());
        exit;
    }
 
    // Remove comments metabox from dashboard
    remove_meta_box('dashboard_recent_comments', 'dashboard', 'normal');
    remove_meta_box('dashboard_quick_press', 'dashboard', 'normal');
 
    // Disable support for comments and trackbacks in post types
    foreach (get_post_types() as $post_type) {
        if (post_type_supports($post_type, 'comments')) {
            remove_post_type_support($post_type, 'comments');
            remove_post_type_support($post_type, 'trackbacks');
        }
    }
});

function my_admin_bar_render() {
    global $wp_admin_bar;
    $wp_admin_bar->remove_menu('comments');
    $wp_admin_bar->remove_menu('new-content');
}
add_action( 'wp_before_admin_bar_render', 'my_admin_bar_render' );
 
// Close comments on the front-end
add_filter('comments_open', '__return_false', 20, 2);
add_filter('pings_open', '__return_false', 20, 2);
 
// Hide existing comments
add_filter('comments_array', '__return_empty_array', 10, 2);
 
// Remove comments page in menu
add_action('admin_menu', function () {
    remove_menu_page('edit-comments.php');
});
 
// Remove comments links from admin bar
add_action('init', function () {
    if (is_admin_bar_showing()) {
        remove_action('admin_bar_menu', 'wp_admin_bar_comments_menu', 60);
    }
});

/**
 * Disable Blog & Posts Completely
*/ 
function disable_blog_features() {
    // Remove the 'Posts' menu from the admin sidebar
    remove_menu_page('edit.php');

    // Disable the 'post' post-type
    remove_post_type_support('post', 'title');
    remove_post_type_support('post', 'editor');
    remove_post_type_support('post', 'author');
    remove_post_type_support('post', 'thumbnail');
    remove_post_type_support('post', 'excerpt');
    remove_post_type_support('post', 'trackbacks');
    remove_post_type_support('post', 'custom-fields');
    remove_post_type_support('post', 'comments');
    remove_post_type_support('post', 'revisions');
    remove_post_type_support('post', 'page-attributes');
    remove_post_type_support('post', 'post-formats');

    // Redirect any attempts to access the 'post' post-type
    global $pagenow;
    if ($pagenow === 'edit.php' && isset($_GET['post_type']) && $_GET['post_type'] === 'post') {
        wp_redirect(admin_url());
        exit;
    }

    // Disable the 'post' post-type from the REST API
    add_filter('rest_endpoints', function ($endpoints) {
        foreach ($endpoints as $route => $endpoint) {
            if (strpos($route, '/posts') !== false) {
                unset($endpoints[$route]);
            }
        }
        return $endpoints;
    });

    // Disable the 'post' post-type from the sitemap
    add_filter('wp_sitemaps_post_types', function ($post_types) {
        unset($post_types['post']);
        return $post_types;
    });

    // Disable the 'post' post-type from the admin bar
    add_action('wp_before_admin_bar_render', function () {
        global $wp_admin_bar;
        $wp_admin_bar->remove_node('new-post');
    });

    // Disable the 'post' post-type from the dashboard activity widget
    add_action('wp_dashboard_setup', function () {
        remove_meta_box('dashboard_activity', 'dashboard', 'normal');
    });

    // Disable the 'post' post-type from the 'At a Glance' dashboard widget
    add_filter('dashboard_glance_items', function ($items) {
        unset($items['post']);
        return $items;
    });
}
add_action('admin_init', 'disable_blog_features');

/**
 * Disable Gutenberg editor & enable the Classic Editor
*/ 
function disable_gutenberg_editor() {
    // Disable for all post types
    add_filter('use_block_editor_for_post', '__return_false', 10);
    add_filter('use_block_editor_for_post_type', '__return_false', 10);
    
    // Bring back the old meta boxes
    add_filter('gutenberg_use_widgets_block_editor', '__return_false');
    add_filter('use_widgets_block_editor', '__return_false');
}
add_action('init', 'disable_gutenberg_editor');
