<?php
/**
 * Sign-up Form AJAX Handlers
 * Handles multi-step sign-up form submissions
 */

// AJAX handlers for sign-up form
add_action('wp_ajax_save_user_type', 'etiuse_save_user_type');
add_action('wp_ajax_save_user_info', 'etiuse_save_user_info');
add_action('wp_ajax_save_user_address', 'etiuse_save_user_address');

/**
 * Validate that a string contains only Persian letters (and optional whitespace)
 *
 * @param string $text
 *
 * @return bool
 */
function etiuse_validate_persian_text($text) {
    $text = trim($text);

    if ($text === '') {
        return false;
    }

    $normalized = preg_replace('/\s+/u', '', $text);
    $normalized = str_replace("\u{200C}", '', $normalized);

    if ($normalized === '') {
        return false;
    }

    return (bool) preg_match('/^(?!.*[\p{N}\p{P}])[\p{Arabic}\s\x{200C}]+$/u', $text);
}

/**
 * Save user type (Step 1)
 */
function etiuse_save_user_type() {
    check_ajax_referer('save_user_type', 'nonce');
    
    $user_id = intval($_POST['user_id']);
    $user_type = sanitize_text_field($_POST['user_type']);
    
    if (!in_array($user_type, ['Clinic', 'Lab'])) {
        wp_send_json_error(['message' => 'نوع کاربر نامعتبر است.']);
    }
    
    update_user_meta($user_id, 'user_type', $user_type);
    
    wp_send_json_success([
        'message' => 'نوع کاربری با موفقیت ذخیره شد.'
    ]);
}

/**
 * Save user personal information (Step 2)
 */
function etiuse_save_user_info() {
    check_ajax_referer('save_user_info', 'nonce');
    
    $user_id = intval($_POST['user_id']);
    $first_name = sanitize_text_field($_POST['first_name']);
    $last_name = sanitize_text_field($_POST['last_name']);
    $organization_name = isset($_POST['organization_name']) ? sanitize_text_field($_POST['organization_name']) : '';
    
    if (empty($first_name) || empty($last_name)) {
        wp_send_json_error(['message' => 'نام و نام خانوادگی اجباری است.']);
    }

    if (!etiuse_validate_persian_text($first_name)) {
        wp_send_json_error(['message' => 'لطفاً نام را فقط با حروف فارسی وارد کنید.']);
    }

    if (!etiuse_validate_persian_text($last_name)) {
        wp_send_json_error(['message' => 'لطفاً نام خانوادگی را فقط با حروف فارسی وارد کنید.']);
    }

    if (!empty($organization_name) && !etiuse_validate_persian_text($organization_name)) {
        wp_send_json_error(['message' => 'نام سازمان باید فقط شامل حروف فارسی باشد.']);
    }
    
    // Update WordPress user fields
    wp_update_user([
        'ID' => $user_id,
        'first_name' => $first_name,
        'last_name' => $last_name
    ]);
    
    // Save organization name if provided
    if (!empty($organization_name)) {
        update_user_meta($user_id, 'user_organization_name', $organization_name);
    }
    
    wp_send_json_success([
        'message' => 'اطلاعات شخصی با موفقیت ذخیره شد.'
    ]);
}

/**
 * Save user address information (Step 3)
 */
function etiuse_save_user_address() {
    check_ajax_referer('save_user_address', 'nonce');
    
    $user_id = intval($_POST['user_id']);
    // Use wp_unslash to handle any slashes and don't sanitize to preserve JSON structure
    $user_address_json = wp_unslash($_POST['user_address']);
    $update_role = isset($_POST['update_role']) && $_POST['update_role'] === 'true';
    
    // Decode the JSON address data
    $address_data = json_decode($user_address_json, true);
    
    // Check if JSON decode was successful
    if (json_last_error() !== JSON_ERROR_NONE) {
        wp_send_json_error(['message' => 'خطا در پردازش داده‌های آدرس: ' . json_last_error_msg()]);
    }
    
    if (!is_array($address_data) || empty($address_data['address'])) {
        wp_send_json_error(['message' => 'آدرس اجباری است.']);
    }
    
    // URL decode the address data to handle encoded characters
    $address_data['address'] = urldecode($address_data['address']);
    $address_data['details'] = urldecode($address_data['details'] ?? '');
    $address_data['notes'] = urldecode($address_data['notes'] ?? '');
    
    // Handle geo_coordinates if it's a JSON string
    if (isset($address_data['geo_coordinates']) && is_string($address_data['geo_coordinates'])) {
        $geo_coords = json_decode($address_data['geo_coordinates'], true);
        if ($geo_coords) {
            $address_data['geo_coordinates'] = $geo_coords;
        }
    }
    
    // Check if address is a valid address (not loading/error messages)
    $address = trim($address_data['address']);
    $invalid_messages = [
        'در حال دریافت آدرس...',
        'آدرس یافت نشد',
        'خطا در دریافت آدرس'
    ];
    
    if (in_array($address, $invalid_messages)) {
        wp_send_json_error(['message' => 'لطفاً موقعیت خود را روی نقشه انتخاب کنید تا آدرس دریافت شود.']);
    }
    
    // Save address information as JSON (with URL-decoded data)
    update_user_meta($user_id, 'user_address', json_encode($address_data));
    
    // Update user role to Client if requested
    if ($update_role) {
        $user = new WP_User($user_id);
        
        // Check if user already has Client role
        if (!in_array('Client', $user->roles)) {
            $user->set_role('Client');
        }
    }
    
    wp_send_json_success([
        'message' => 'آدرس با موفقیت ذخیره شد.'
    ]);
}
