/**
 * Dashboard JavaScript Functionality
 * Handles dashboard navigation, section switching, and header updates
 */

document.addEventListener('DOMContentLoaded', function() {


    let currentSection = 'dashboard';
    const sections = ['dashboard', 'new-order', 'orders', 'charges'];

    // Initialize dashboard
    function initDashboard() {
        // Check URL hash on page load
        const hash = window.location.hash.substring(1);
        if (hash && sections.includes(hash)) {
            loadSection(hash);
        } else {
            loadSection('dashboard');
        }

        // Add click event listeners to navigation
        document.querySelectorAll('.nav-link[data-section]').forEach(link => {
            link.addEventListener('click', function(e) {
                e.preventDefault();
                const section = this.getAttribute('data-section');
                loadSection(section);
            });
        });

        // Handle browser back/forward buttons
        window.addEventListener('popstate', function() {
            const hash = window.location.hash.substring(1);
            if (hash && sections.includes(hash)) {
                loadSection(hash, false);
            } else {
                loadSection('dashboard', false);
            }
        });


        // Initialize sidebar toggle
        initSidebarToggle();
        
        // Initialize settings dropdown
        initSettingsDropdown();
    }

    // Load section content
    function loadSection(section, updateHistory = true) {
        if (currentSection === section) return;

        // Update navigation
        updateNavigation(section);

        // Update content
        updateContent(section);

        // Update URL and browser history
        if (updateHistory) {
            window.history.pushState({}, '', `#${section}`);
        }

        currentSection = section;
    }

    // Update navigation active state
    function updateNavigation(section) {
        document.querySelectorAll('.nav-link').forEach(link => {
            link.classList.remove('active');
        });
        
        const activeLink = document.querySelector(`[data-section="${section}"]`);
        if (activeLink) {
            activeLink.classList.add('active');
        }
    }

    // Update content sections
    function updateContent(section) {
        document.querySelectorAll('.content-section').forEach(sectionEl => {
            sectionEl.classList.remove('active');
        });
        
        const targetSection = document.getElementById(`${section}-section`);
        if (targetSection) {
            targetSection.classList.add('active');
        }
    }


    // Initialize sidebar toggle functionality
    function initSidebarToggle() {
        const sidebarToggle = document.getElementById('sidebar-toggle');
        const sidebar = document.getElementById('dashboard-sidebar');
        const overlay = document.getElementById('sidebar-overlay');
        
        if (sidebarToggle && sidebar) {
            sidebarToggle.addEventListener('click', function() {
                const isMobile = window.innerWidth <= 1024;
                
                if (isMobile) {
                    // Mobile behavior: toggle open/close with overlay
                    sidebar.classList.toggle('open');
                    if (overlay) {
                        overlay.classList.toggle('show');
                    }
                } else {
                    // Desktop behavior: toggle collapsed/expanded
                    sidebar.classList.toggle('collapsed');
                    
                    // Store state in localStorage
                    const isCollapsed = sidebar.classList.contains('collapsed');
                    localStorage.setItem('sidebar-collapsed', isCollapsed);
                }
            });
            
            // Close sidebar when clicking overlay (mobile only)
            if (overlay) {
                overlay.addEventListener('click', function() {
                    sidebar.classList.remove('open');
                    overlay.classList.remove('show');
                });
            }
            
            // Close sidebar when clicking outside (mobile only)
            document.addEventListener('click', function(e) {
                const isMobile = window.innerWidth <= 1024;
                if (isMobile && sidebar.classList.contains('open')) {
                    if (!sidebar.contains(e.target) && !sidebarToggle.contains(e.target)) {
                        sidebar.classList.remove('open');
                        if (overlay) {
                            overlay.classList.remove('show');
                        }
                    }
                }
            });
            
            // Handle window resize
            window.addEventListener('resize', function() {
                const isMobile = window.innerWidth <= 1024;
                if (!isMobile) {
                    // Desktop: restore collapsed state
                    sidebar.classList.remove('open');
                    if (overlay) {
                        overlay.classList.remove('show');
                    }
                    const isCollapsed = localStorage.getItem('sidebar-collapsed') === 'true';
                    if (isCollapsed) {
                        sidebar.classList.add('collapsed');
                    } else {
                        sidebar.classList.remove('collapsed');
                    }
                } else {
                    // Mobile: remove collapsed state, use open state instead
                    sidebar.classList.remove('collapsed');
                }
            });
            
            // Restore state from localStorage (desktop only)
            const isMobile = window.innerWidth <= 1024;
            if (!isMobile) {
                const isCollapsed = localStorage.getItem('sidebar-collapsed') === 'true';
                if (isCollapsed) {
                    sidebar.classList.add('collapsed');
                }
            }
        }
    }
    
    // Initialize settings dropdown functionality
    function initSettingsDropdown() {
        const settingsToggle = document.getElementById('settings-toggle');
        const settingsDropdown = document.getElementById('settings-dropdown');
        
        if (settingsToggle && settingsDropdown) {
            settingsToggle.addEventListener('click', function(e) {
                e.stopPropagation();
                settingsDropdown.classList.toggle('show');
            });
            
            // Close dropdown when clicking outside
            document.addEventListener('click', function(e) {
                if (!settingsToggle.contains(e.target) && !settingsDropdown.contains(e.target)) {
                    settingsDropdown.classList.remove('show');
                }
            });
            
            // Close dropdown when clicking on dropdown items
            settingsDropdown.addEventListener('click', function(e) {
                if (e.target.closest('.dropdown-item')) {
                    settingsDropdown.classList.remove('show');
                }
            });
        }
    }

    // Initialize the dashboard
    initDashboard();
});
