<?php

if ( ! defined( 'ETIUSE_SMS_API_URL' ) ) {
    define( 'ETIUSE_SMS_API_URL', 'https://console.melipayamak.com/api/send/shared/41aa7a27058e4d709ec51039af845595' );
}

if ( ! function_exists( 'etiuse_send_sms' ) ) {
    /**
     * Send SMS via the Melipayamak shared body API.
     *
     * @param string $phone
     * @param int    $body_id
     * @param array  $args
     * @return string|false
     */
    function etiuse_send_sms( $phone, $body_id, $args = array() ) {
        $data        = array(
            'bodyId' => $body_id,
            'to'     => $phone,
            'args'   => array_map( 'strval', $args ),
        );
        $data_string = wp_json_encode( $data );

        $ch = curl_init( ETIUSE_SMS_API_URL );
        curl_setopt( $ch, CURLOPT_CUSTOMREQUEST, 'POST' );
        curl_setopt( $ch, CURLOPT_POSTFIELDS, $data_string );
        curl_setopt( $ch, CURLOPT_SSL_VERIFYPEER, false );
        curl_setopt( $ch, CURLOPT_RETURNTRANSFER, true );
        curl_setopt(
            $ch,
            CURLOPT_HTTPHEADER,
            array(
                'Content-Type: application/json',
                'Content-Length: ' . strlen( $data_string ),
            )
        );

        $result = curl_exec( $ch );
        curl_close( $ch );

        return $result;
    }
}

/**
 * Get translation mappings for database values to Persian
 * 
 * This function returns all available translation mappings for different value types.
 * The system is expandable - add new mappings by adding new keys to the returned array.
 * 
 * @return array Associative array of translation mappings
 */
if ( ! function_exists( 'etiuse_get_translation_mappings' ) ) {
    function etiuse_get_translation_mappings() {
        return array(
            // Service Types
            'service_type' => array(
                'orthodontics'          => 'ارتودنسی',
                'fixed_prosthesis'      => 'پروتز ثابت',
                'removable_prosthesis'  => 'پروتز متحرک',
            ),
            
            // Sub-services (all sub-service values)
            'sub_services' => array(
                // Orthodontics
                'clear_aligner'             => 'ارتودنسی شفاف',
                'orthodontic_retainer'      => 'پلاک ارتودنسی',
                'orthognathic_surgery'      => 'ارتو سرجری',
                'diagnosis_analysis'        => 'آنالیز و تشخیص',
                
                // Fixed Prosthesis - Main
                'crown'                     => 'روکش',
                'abutment'                  => 'اباتمنت',
                'laminate'                  => 'لمینت',
                'substructure'              => 'ساب استراکچر فیکس',
                'composite_restoration'     => 'ایندایرکت کامپوزیت رستوریشن',
                'smile_design'              => 'طراحی لبخند',
                
                // Crown Types
                'pfm'                       => 'PFM',
                'zirconia'                  => 'زیرکونیا',
                'lithium_disilicate'        => 'لیتیوم دی سیلکات',
                'pmma'                      => 'PMMA',
                
                // Zirconia Types
                'implant'                   => 'ایمپلنت',
                'dental'                    => 'دندانی',
                
                // PMMA Types
                'temporary_implant'         => 'ایمپلنت موقت',
                'temporary'                 => 'موقت',
                
                // Abutment Types
                'preparation'               => 'تراش',
                'customized'                => 'کاستومایزد',
                'abutment_selection'        => 'انتخاب اباتمنت',
                'types_of_jig'              => 'انواع جیگ',
                
                // Laminate Types
                'core_laminate'             => 'کور لمینت',
                
                // Laminate Laminate Types
                'feldspathic'               => 'فلدسپاتیک',
                'ips'                       => 'ips',
                
                // Removable Prosthesis - Main
                'denture'                   => 'دنچر',
                'partial'                   => 'پارسیل',
                
                // Denture Types
                'obturator'                 => 'آبچوراتور',
                'overdenture'               => 'اوردنچر',
                
                // Partial Types
                'acrylic'                   => 'آکریلی',
                'metal'                     => 'فلزی',
                
                // Acrylic Types
                'hard'                      => 'سخت',
                'soft'                      => 'نرم',
            ),
            
            // Extras (Removable Services)
            'extras' => array(
                'bar'                      => 'بار',
                'wing'                     => 'بال',
                'night_guard'              => 'نایت گارد',
            ),
            
            // Status
            'status' => array(
                'pending'                  => 'در انتظار تأیید',
                'in_progress'              => 'در حال انجام',
                'completed'                => 'تکمیل شده',
                'cancelled'                => 'لغو شده',
            ),
            
            // Patient Age
            'patient_age' => array(
                'child'                    => 'کودک',
                'teen'                     => 'نوجوان',
                'adult'                    => 'بزرگسال',
            ),
        );
    }
}

/**
 * Translate a single value from English to Persian
 * 
 * @param string $value The English value to translate
 * @param string $type The type of mapping to use (service_type, sub_services, extras, status, etc.)
 * @param string $fallback Optional fallback value if translation not found. If not provided, returns original value
 * @return string The Persian translation or fallback/original value
 */
if ( ! function_exists( 'etiuse_translate_value' ) ) {
    function etiuse_translate_value( $value, $type, $fallback = null ) {
        if ( empty( $value ) ) {
            return $fallback !== null ? $fallback : '';
        }
        
        $mappings = etiuse_get_translation_mappings();
        
        if ( ! isset( $mappings[ $type ] ) ) {
            return $fallback !== null ? $fallback : $value;
        }
        
        $translation = isset( $mappings[ $type ][ $value ] ) 
            ? $mappings[ $type ][ $value ] 
            : ( $fallback !== null ? $fallback : $value );
        
        return $translation;
    }
}

/**
 * Translate multiple values (array) from English to Persian
 * 
 * @param array|string $values The values to translate (can be array or JSON string)
 * @param string $type The type of mapping to use (service_type, sub_services, extras, status, etc.)
 * @return array|string Array of translated values, or empty array if input is invalid
 */
if ( ! function_exists( 'etiuse_translate_values' ) ) {
    function etiuse_translate_values( $values, $type ) {
        // Handle JSON string input
        if ( is_string( $values ) ) {
            $decoded = json_decode( $values, true );
            if ( json_last_error() === JSON_ERROR_NONE && is_array( $decoded ) ) {
                $values = $decoded;
            } else {
                // If it's not valid JSON, try to decode with stripslashes (for escaped JSON)
                $clean_json = stripslashes( $values );
                $decoded = json_decode( $clean_json, true );
                if ( json_last_error() === JSON_ERROR_NONE && is_array( $decoded ) ) {
                    $values = $decoded;
                } else {
                    return array();
                }
            }
        }
        
        // Handle non-array input
        if ( ! is_array( $values ) ) {
            return array();
        }
        
        // Translate each value
        $translated = array();
        foreach ( $values as $value ) {
            $translated[] = etiuse_translate_value( $value, $type );
        }
        
        return $translated;
    }
}

/**
 * Format status display (backward compatibility wrapper)
 * This function maintains compatibility with existing code that uses format_status_display()
 * 
 * @param string $status The status value to translate
 * @return string The Persian status name
 */
if ( ! function_exists( 'format_status_display' ) ) {
    function format_status_display( $status ) {
        return etiuse_translate_value( $status, 'status', $status );
    }
}


