/**
 * Dental File Upload Component
 * Reusable file upload component with progress tracking and validation
 * Can be used anywhere in the application
 */

class DentalFileUpload {
    constructor(options = {}) {
        this.options = {
            // Upload settings
            maxFiles: 15,
            maxFileSize: 200 * 1024 * 1024, // 200 MB
            allowedTypes: {
                // Images
                'image/png': '.png',
                'image/jpeg': '.jpg,.jpeg',
                'image/webp': '.webp',
                'image/heic': '.heic',
                'image/heif': '.heif',
                'image/heic-sequence': '.heic',
                'image/heif-sequence': '.heif',
                // Videos
                'video/mp4': '.mp4',
                'video/webm': '.webm',
                'video/quicktime': '.mov',
                'video/x-msvideo': '.avi',
                'video/x-matroska': '.mkv',
                // Audio
                'audio/mpeg': '.mp3',
                'audio/wav': '.wav',
                'audio/ogg': '.ogg',
                'audio/mp4': '.m4a',
                'audio/aac': '.aac',
                'audio/flac': '.flac',
                // Archives
                'application/zip': '.zip',
                'application/x-zip-compressed': '.zip',
                'application/x-zip': '.zip',
                'application/zip-compressed': '.zip',
                'application/octet-stream': '.zip,.rar,.7z,.tar,.gz', // Fallback for generic detection
                'application/vnd.rar': '.rar',
                'application/x-rar-compressed': '.rar',
                'application/x-7z-compressed': '.7z',
                'application/x-tar': '.tar',
                'application/gzip': '.gz',
                // 3D & CAD
                'application/vnd.ms-pki.stl': '.stl',
                'application/sla': '.stl',
                'application/x-navistyle': '.stl',
                'text/vnd.fly': '.fly',
                // Documents
                'application/pdf': '.pdf',
                'application/msword': '.doc',
                'application/vnd.openxmlformats-officedocument.wordprocessingml.document': '.docx',
                'application/vnd.ms-excel': '.xls',
                'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet': '.xlsx',
                'application/vnd.ms-powerpoint': '.ppt',
                'application/vnd.openxmlformats-officedocument.presentationml.presentation': '.pptx',
                'text/csv': '.csv',
                'text/plain': '.txt',
                'application/rtf': '.rtf'
            },
            
            // API endpoints
            uploadUrl: window.dentalUploadAjaxUrl || '/wp-admin/admin-ajax.php',
            uploadAction: 'upload_dental_file',
            deleteAction: 'delete_dental_file',
            
            // UI settings
            showProgress: true,
            showFileList: true,
            showFormatInfo: true,
            dragDropEnabled: true,
            
            // Callbacks
            onFileAdded: null,
            onFileRemoved: null,
            onUploadComplete: null,
            onUploadError: null,
            onValidationError: null,
            
            // Override defaults with provided options
            ...options
        };
        
        this.selectedFiles = [];
        this.uploadInProgress = false;
        this.isFileDialogOpen = false; // Flag to prevent multiple file dialogs
        this.element = null;
        this.uploadArea = null;
        this.fileInput = null;
        this.fileList = null;
        this.progressContainer = null;
        
        this.init();
    }
    
    /**
     * Initialize the upload component
     */
    init() {
        this.setupElementReferences();
        this.setupEventListeners();
        this.setupDragAndDrop();
    }
    
    /**
     * Setup references to existing HTML elements
     */
    setupElementReferences() {
        // Find the existing upload container
        this.element = document.getElementById('dental-file-upload');
        
        if (!this.element) {
            return;
        }
        
        // Store component reference for event handlers
        this.element.__component = this;
        
        // Store references to important elements
        this.uploadArea = document.getElementById('dental-upload-area');
        this.fileInput = document.getElementById('dental-file-input');
        this.fileList = document.getElementById('dental-file-list-items');
        this.progressContainer = document.getElementById('dental-progress');
        this.progressList = document.getElementById('dental-progress-list');
        this.fileListContainer = document.getElementById('dental-file-list');
        this.fileCount = document.getElementById('dental-file-count');
        this.selectBtn = document.getElementById('dental-select-btn');
        this.clearBtn = document.getElementById('dental-clear-btn');
        this.cancelBtn = document.getElementById('dental-cancel-btn');
    }
    
    /**
     * Setup event listeners
     */
    setupEventListeners() {
        // Prevent multiple event listener attachments
        if (this.eventListenersAttached) {
            return;
        }
        
        // File input change
        if (this.fileInput) {
            this.fileInput.addEventListener('change', (e) => {
                this.isFileDialogOpen = false; // Reset flag when dialog closes
                
                if (e.target.files.length > 0) {
                    this.handleFileSelection(e.target.files);
                }
                // Reset input after a short delay to prevent immediate reopening
                setTimeout(() => {
                    e.target.value = '';
                }, 100);
            });
        }
        
        // Select files button
        if (this.selectBtn) {
            this.selectBtn.addEventListener('click', (e) => {
                e.preventDefault();
                e.stopPropagation();
                
                // Prevent multiple file dialogs from opening
                if (this.isFileDialogOpen) {
                    return;
                }
                
                this.isFileDialogOpen = true; // Set flag to prevent multiple dialogs
                
                // Clear the input first to ensure clean state
                this.fileInput.value = '';
                // Use setTimeout to ensure the click happens after the value is cleared
                setTimeout(() => {
                    this.fileInput.click();
                }, 10);
                
                // Fallback: Reset flag after 5 seconds in case dialog doesn't trigger change event
                setTimeout(() => {
                    if (this.isFileDialogOpen) {
                        this.isFileDialogOpen = false;
                    }
                }, 5000);
            });
        }
        
        // Clear all files
        if (this.clearBtn) {
            this.clearBtn.addEventListener('click', () => {
                this.clearAllFiles();
            });
        }
        
        // Cancel upload
        if (this.cancelBtn) {
            this.cancelBtn.addEventListener('click', () => {
                this.cancelAllUploads();
            });
        }
        
        // Mark event listeners as attached
        this.eventListenersAttached = true;
    }
    
    /**
     * Setup drag and drop functionality
     */
    setupDragAndDrop() {
        // Prevent multiple drag and drop setup
        if (this.dragDropAttached) {
            return;
        }
        
        if (!this.options.dragDropEnabled) {
            return;
        }
        
        if (!this.uploadArea) {
            return;
        }
        
        this.uploadArea.addEventListener('dragover', (e) => {
            e.preventDefault();
            this.uploadArea.classList.add('drag-over');
        });
        
        this.uploadArea.addEventListener('dragleave', (e) => {
            e.preventDefault();
            if (!this.uploadArea.contains(e.relatedTarget)) {
                this.uploadArea.classList.remove('drag-over');
            }
        });
        
        this.uploadArea.addEventListener('drop', (e) => {
            e.preventDefault();
            this.uploadArea.classList.remove('drag-over');
            
            const files = e.dataTransfer.files;
            if (files.length > 0) {
                this.handleFileSelection(files);
            }
        });
        
        this.uploadArea.addEventListener('dragenter', (e) => {
            e.preventDefault();
        });
        
        // Mark drag and drop as attached
        this.dragDropAttached = true;
    }
    
    /**
     * Handle file selection and start upload
     */
    handleFileSelection(files) {
        const fileArray = Array.from(files);
        
        // Check total file count
        if (this.selectedFiles.length + fileArray.length > this.options.maxFiles) {
            this.showError(`حداکثر ${this.options.maxFiles} فایل مجاز است. شما ${this.selectedFiles.length} فایل انتخاب کرده‌اید.`);
            return;
        }
        
        // Process each file
        fileArray.forEach(file => {
            this.uploadFile(file);
        });
    }
    
    /**
     * Upload file with progress tracking
     */
    uploadFile(file) {
        const validation = this.validateFile(file);
        
        if (!validation.isValid) {
            // Show error but don't upload invalid files
            this.addFileToList(file, validation);
            this.callCallback('onValidationError', file, validation.errors);
            return;
        }
        
        // Add file to list immediately with progress bar
        const fileItem = this.createFileItem(file, validation);
        this.fileList.appendChild(fileItem);
        
        // Show file list container
        this.fileListContainer.style.display = 'block';
        
        // Create progress bar within the file item
        const progressBar = fileItem.querySelector('.file-progress-bar');
        const progressFill = fileItem.querySelector('.file-progress-fill');
        const progressText = fileItem.querySelector('.file-progress-text');
        
        // Show progress bar
        progressBar.style.display = 'block';
        
        // Create FormData
        const formData = new FormData();
        formData.append('file', file);
        formData.append('action', this.options.uploadAction);
        formData.append('user_id', this.getCurrentUserId());
        formData.append('nonce', this.getNonce('dental_file_upload'));
        
        // Add dental_id if available (for permanent uploads)
        if (this.options.dentalId) {
            formData.append('dental_id', this.options.dentalId);
        }
        
        // Create XMLHttpRequest for progress tracking
        const xhr = new XMLHttpRequest();
        
        // Track upload progress
        xhr.upload.addEventListener('progress', (e) => {
            if (e.lengthComputable) {
                const percentComplete = (e.loaded / e.total) * 100;
                this.updateFileProgressBar(progressFill, progressText, percentComplete);
            }
        });
        
        // Handle upload completion
        xhr.addEventListener('load', () => {
            if (xhr.status === 200) {
                try {
                    const response = JSON.parse(xhr.responseText);
                    
                    if (response.success) {
                        // Upload successful
                        this.completeFileUpload(fileItem, file, response.data);
                        this.callCallback('onUploadComplete', file, response.data);
                    } else {
                        // Upload failed
                        this.handleFileUploadError(fileItem, file, response.message || 'خطا در آپلود فایل');
                        this.callCallback('onUploadError', file, response.message);
                    }
                } catch (e) {
                    this.handleFileUploadError(fileItem, file, 'خطا در پردازش پاسخ سرور');
                    this.callCallback('onUploadError', file, 'خطا در پردازش پاسخ سرور');
                }
            } else {
                this.handleFileUploadError(fileItem, file, `خطای سرور: ${xhr.status}`);
                this.callCallback('onUploadError', file, `خطای سرور: ${xhr.status}`);
            }
        });
        
        // Handle upload error
        xhr.addEventListener('error', () => {
            this.handleFileUploadError(fileItem, file, 'خطا در اتصال به سرور');
            this.callCallback('onUploadError', file, 'خطا در اتصال به سرور');
        });
        
        // Handle upload abort
        xhr.addEventListener('abort', () => {
            this.handleFileUploadAbort(fileItem, file);
        });
        
        // Start upload
        xhr.open('POST', this.options.uploadUrl, true);
        xhr.send(formData);
        
        // Store xhr reference for potential cancellation
        fileItem.dataset.xhr = xhr;
    }
    
    /**
     * Create progress item element
     */
    createProgressItem(file) {
        const progressItem = document.createElement('div');
        progressItem.className = 'progress-item';
        progressItem.dataset.fileName = file.name;
        
        const fileTypeCategory = this.getFileTypeCategory(file.type);
        const fileSize = this.formatFileSize(file.size);
        
        progressItem.innerHTML = `
            <div class="file-item-icon ${fileTypeCategory}">
                <i class="bi bi-${this.getFileIcon(fileTypeCategory)}"></i>
            </div>
            <div class="progress-item-info">
                <div class="progress-item-name">${file.name}</div>
                <div class="progress-item-size">${fileSize}</div>
                <div class="progress-bar-item">
                    <div class="progress-bar-fill" style="width: 0%"></div>
                </div>
            </div>
            <div class="progress-item-status uploading">در حال آپلود...</div>
        `;
        
        return progressItem;
    }
    
    /**
     * Update file progress bar
     */
    updateFileProgressBar(progressFill, progressText, percentComplete) {
        if (progressFill) {
            progressFill.style.width = percentComplete + '%';
        }
        
        if (progressText) {
            progressText.textContent = Math.round(percentComplete) + '%';
        }
    }
    
    /**
     * Update progress bar (legacy method)
     */
    updateProgressBar(progressItem, percentComplete) {
        const progressBar = progressItem.querySelector('.progress-bar-fill');
        const statusElement = progressItem.querySelector('.progress-item-status');
        
        progressBar.style.width = percentComplete + '%';
        statusElement.textContent = `${Math.round(percentComplete)}%`;
    }
    
    /**
     * Complete file upload successfully
     */
    completeFileUpload(fileItem, file, uploadData) {
        const statusElement = fileItem.querySelector('.file-item-status');
        const progressBar = fileItem.querySelector('.file-progress-bar');
        
        // Update status
        statusElement.textContent = 'آپلود شد';
        statusElement.className = 'file-item-status uploaded';
        
        // Hide progress bar
        progressBar.style.display = 'none';
        
        // Add to selected files with server data
        const fileData = {
            file: file,
            uploadData: uploadData,
            fileItem: fileItem
        };
        this.selectedFiles.push(fileData);
        
        this.updateFileCount();
        this.callCallback('onFileAdded', file, uploadData);
    }
    
    /**
     * Complete upload successfully (legacy method)
     */
    completeUpload(progressItem, file, uploadData) {
        const statusElement = progressItem.querySelector('.progress-item-status');
        const progressBar = progressItem.querySelector('.progress-bar-fill');
        
        // Update status
        statusElement.textContent = 'آپلود شد';
        statusElement.className = 'progress-item-status completed';
        progressBar.classList.add('completed');
        
        // Add to selected files with server data
        const fileData = {
            file: file,
            uploadData: uploadData,
            progressItem: progressItem
        };
        this.selectedFiles.push(fileData);
        
        // Move to file list after a short delay
        setTimeout(() => {
            progressItem.remove();
            
            // Create file list item
            const fileItem = this.createFileItem(file, { isValid: true }, uploadData);
            this.fileList.appendChild(fileItem);
            
            // Hide progress container if no more uploads
            if (this.progressList.children.length === 0) {
                this.progressContainer.style.display = 'none';
                this.uploadArea.classList.remove('uploading');
            }
            
            this.updateFileCount();
            this.callCallback('onFileAdded', file, uploadData);
        }, 1000);
    }
    
    /**
     * Handle file upload error
     */
    handleFileUploadError(fileItem, file, errorMessage) {
        const statusElement = fileItem.querySelector('.file-item-status');
        const progressBar = fileItem.querySelector('.file-progress-bar');
        
        // Update status
        statusElement.textContent = 'خطا';
        statusElement.className = 'file-item-status error';
        
        // Hide progress bar
        progressBar.style.display = 'none';
        
        // Add error message
        const errorDiv = document.createElement('div');
        errorDiv.className = 'file-item-error';
        errorDiv.textContent = errorMessage;
        fileItem.querySelector('.file-item-info').appendChild(errorDiv);
        
        // Remove file item after delay
        setTimeout(() => {
            fileItem.remove();
            this.updateFileCount();
        }, 3000);
    }
    
    /**
     * Handle upload error (legacy method)
     */
    handleUploadError(progressItem, file, errorMessage) {
        const statusElement = progressItem.querySelector('.progress-item-status');
        const progressBar = progressItem.querySelector('.progress-bar-fill');
        
        // Update status
        statusElement.textContent = 'خطا';
        statusElement.className = 'progress-item-status error';
        progressBar.classList.add('error');
        
        // Add error message
        const errorDiv = document.createElement('div');
        errorDiv.className = 'file-item-error';
        errorDiv.textContent = errorMessage;
        progressItem.querySelector('.progress-item-info').appendChild(errorDiv);
        
        // Remove from progress after delay
        setTimeout(() => {
            progressItem.remove();
            
            // Hide progress container if no more uploads
            if (this.progressList.children.length === 0) {
                this.progressContainer.style.display = 'none';
                this.uploadArea.classList.remove('uploading');
            }
        }, 3000);
    }
    
    /**
     * Handle file upload abort
     */
    handleFileUploadAbort(fileItem, file) {
        const statusElement = fileItem.querySelector('.file-item-status');
        
        statusElement.textContent = 'لغو شد';
        statusElement.className = 'file-item-status error';
        
        // Remove file item after delay
        setTimeout(() => {
            fileItem.remove();
            this.updateFileCount();
        }, 2000);
    }
    
    /**
     * Handle upload abort (legacy method)
     */
    handleUploadAbort(progressItem, file) {
        const statusElement = progressItem.querySelector('.progress-item-status');
        
        statusElement.textContent = 'لغو شد';
        statusElement.className = 'progress-item-status error';
        
        // Remove from progress after delay
        setTimeout(() => {
            progressItem.remove();
            
            // Hide progress container if no more uploads
            if (this.progressList.children.length === 0) {
                this.progressContainer.style.display = 'none';
                this.uploadArea.classList.remove('uploading');
            }
        }, 2000);
    }
    
    /**
     * Create file item element
     */
    createFileItem(file, validation, uploadData = null) {
        const fileItem = document.createElement('div');
        fileItem.className = `file-item ${validation.isValid ? 'success' : 'error'}`;
        fileItem.dataset.fileName = file.name;
        if (uploadData && uploadData.file_id) {
            fileItem.dataset.fileId = uploadData.file_id;
        }
        
        const fileTypeCategory = this.getFileTypeCategory(file.type);
        const fileSize = this.formatFileSize(file.size);
        
        fileItem.innerHTML = `
            <div class="file-item-icon ${fileTypeCategory}">
                <i class="bi bi-${this.getFileIcon(fileTypeCategory)}"></i>
            </div>
            <div class="file-item-info">
                <div class="file-item-name">${file.name}</div>
                <div class="file-item-details">
                    <span class="file-item-size">${fileSize}</span>
                    <span class="file-item-status uploading">در حال آپلود...</span>
                </div>
                ${!validation.isValid ? `<div class="file-item-error">${validation.errors.join(', ')}</div>` : ''}
                <div class="file-progress-bar" style="display: none;">
                    <div class="file-progress-fill" style="width: 0%"></div>
                    <span class="file-progress-text">0%</span>
                </div>
            </div>
            <div class="file-item-actions">
                <button type="button" class="btn-remove-file" onclick="this.closest('.dental-file-upload').__component.removeFile('${file.name}')">
                    <i class="bi bi-x"></i>
                </button>
            </div>
        `;
        
        return fileItem;
    }
    
    /**
     * Add file to list (for invalid files)
     */
    addFileToList(file, validation) {
        const fileItem = this.createFileItem(file, validation);
        this.fileList.appendChild(fileItem);
        
        // Show file list container even for invalid files
        this.fileListContainer.style.display = 'block';
        
        if (validation.isValid) {
            this.selectedFiles.push({ file: file });
        }
        
        this.updateFileCount();
    }
    
    /**
     * Remove file from list and server
     */
    removeFile(fileName) {
        // Find file item and file data
        const fileItem = this.fileList.querySelector(`[data-file-name="${fileName}"]`);
        const fileData = this.selectedFiles.find(f => f.file && f.file.name === fileName);
        
        if (fileItem) {
            fileItem.remove();
        }
        
        // Remove from selected files array
        this.selectedFiles = this.selectedFiles.filter(f => !f.file || f.file.name !== fileName);
        
        // Mark as deleted in database if it was uploaded
        if (fileData && fileData.uploadData && fileData.uploadData.attachment_id) {
            this.deleteFileFromServer(fileData.uploadData.attachment_id);
        }
        
        this.updateFileCount();
        this.callCallback('onFileRemoved', fileData ? fileData.file : null);
    }
    
    /**
     * Delete file from server (soft delete - database only)
     */
    deleteFileFromServer(fileId) {
        const formData = new FormData();
        formData.append('action', this.options.deleteAction);
        formData.append('attachment_id', fileId);
        formData.append('nonce', this.getNonce('dental_file_delete'));
        
        fetch(this.options.uploadUrl, {
            method: 'POST',
            body: formData
        })
        .then(response => response.json())
        .then(data => {
            // File deletion handled silently
        })
        .catch(error => {
            // Error handling silent
        });
    }
    
    /**
     * Update file count
     */
    updateFileCount() {
        const count = this.selectedFiles.length;
        this.fileCount.textContent = `${count} فایل`;
        
        if (count > 0) {
            this.fileListContainer.style.display = 'block';
        } else {
            this.fileListContainer.style.display = 'none';
        }
    }
    
    /**
     * Clear all files
     */
    clearAllFiles() {
        // Cancel all ongoing uploads
        const progressItems = this.progressList.querySelectorAll('.progress-item');
        progressItems.forEach(item => {
            const xhr = item.dataset.xhr;
            if (xhr) {
                xhr.abort();
            }
        });
        
        // Mark all uploaded files as deleted in database
        this.selectedFiles.forEach(fileData => {
            if (fileData.uploadData && fileData.uploadData.attachment_id) {
                this.deleteFileFromServer(fileData.uploadData.attachment_id);
            }
        });
        
        // Clear UI
        this.fileList.innerHTML = '';
        this.progressList.innerHTML = '';
        this.selectedFiles = [];
        
        // Hide containers
        this.fileListContainer.style.display = 'none';
        this.progressContainer.style.display = 'none';
        this.uploadArea.classList.remove('uploading');
        
        this.updateFileCount();
    }
    
    /**
     * Cancel all uploads
     */
    cancelAllUploads() {
        // Cancel all ongoing uploads
        const progressItems = this.progressList.querySelectorAll('.progress-item');
        progressItems.forEach(item => {
            const xhr = item.dataset.xhr;
            if (xhr) {
                xhr.abort();
            }
        });
        
        // Clear progress list
        this.progressList.innerHTML = '';
        
        // Hide progress container
        this.progressContainer.style.display = 'none';
        this.uploadArea.classList.remove('uploading');
    }
    
    /**
     * Validate file
     */
    validateFile(file) {
        const errors = [];
        
        // Check file size
        if (file.size > this.options.maxFileSize) {
            errors.push(`حجم فایل بیش از ${this.formatFileSize(this.options.maxFileSize)} است`);
        }
        
        // Check file type
        let isTypeValid = false;
        
        // First check if MIME type is directly allowed
        if (this.options.allowedTypes[file.type]) {
            isTypeValid = true;
        } else {
            // Check if it's a generic MIME type but has valid extension
            const fileExtension = file.name.split('.').pop().toLowerCase();
            const validExtensions = ['zip', 'rar', '7z', 'tar', 'gz', 'png', 'jpg', 'jpeg', 'webp', 'heic', 'heif', 'mp4', 'webm', 'mov', 'avi', 'mkv', 'mp3', 'wav', 'ogg', 'm4a', 'aac', 'flac', 'stl', 'fly', 'pdf', 'doc', 'docx', 'xls', 'xlsx', 'ppt', 'pptx', 'csv', 'txt', 'rtf'];
            
            if (file.type === 'application/octet-stream' && validExtensions.includes(fileExtension)) {
                isTypeValid = true;
            }
        }
        
        if (!isTypeValid) {
            errors.push('فرمت فایل پشتیبانی نمی‌شود');
        }
        
        return {
            isValid: errors.length === 0,
            errors: errors
        };
    }
    
    /**
     * Get file type category for icon
     */
    getFileTypeCategory(mimeType) {
        if (mimeType.startsWith('image/')) return 'image';
        if (mimeType.startsWith('video/')) return 'video';
        if (mimeType.startsWith('audio/')) return 'audio';
        if (mimeType.startsWith('application/')) {
            if (['application/zip', 'application/vnd.rar', 'application/x-rar-compressed', 'application/x-7z-compressed', 'application/x-tar', 'application/gzip'].includes(mimeType)) {
                return 'archive';
            }
            if (['application/vnd.ms-pki.stl', 'application/sla', 'application/x-navistyle'].includes(mimeType)) {
                return 'cad';
            }
            return 'document';
        }
        if (mimeType.startsWith('text/')) {
            if (mimeType === 'text/vnd.fly') return 'cad';
            return 'document';
        }
        return 'document';
    }
    
    /**
     * Get file icon based on category
     */
    getFileIcon(category) {
        const icons = {
            'image': 'image',
            'video': 'camera-video',
            'audio': 'music-note',
            'archive': 'archive',
            'cad': 'box',
            'document': 'file-earmark'
        };
        return icons[category] || 'file-earmark';
    }
    
    /**
     * Format file size
     */
    formatFileSize(bytes) {
        if (bytes === 0) return '0 Bytes';
        const k = 1024;
        const sizes = ['Bytes', 'KB', 'MB', 'GB'];
        const i = Math.floor(Math.log(bytes) / Math.log(k));
        return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
    }
    
    /**
     * Get current user ID (WordPress)
     */
    getCurrentUserId() {
        // This should be provided by WordPress
        return window.dentalUploadUserId || 0;
    }
    
    /**
     * Get nonce for security
     */
    getNonce(action) {
        // This should be provided by WordPress
        return window.dentalUploadNonces ? window.dentalUploadNonces[action] : '';
    }
    
    /**
     * Call callback function if provided
     */
    callCallback(callbackName, ...args) {
        if (this.options[callbackName] && typeof this.options[callbackName] === 'function') {
            this.options[callbackName](...args);
        }
    }
    
    /**
     * Show error message
     */
    showError(message) {
        if (this.options.onValidationError) {
            this.options.onValidationError(null, [message]);
        } else {
            alert(message);
        }
    }
    
    /**
     * Get selected files
     */
    getSelectedFiles() {
        return this.selectedFiles;
    }
    
    /**
     * Get uploaded file IDs
     */
    getUploadedFileIds() {
        return this.selectedFiles
            .filter(fileData => fileData.uploadData && fileData.uploadData.attachment_id)
            .map(fileData => fileData.uploadData.attachment_id);
    }
    
    /**
     * Get uploaded files data for form submission
     */
    getUploadedFilesData() {
        return this.selectedFiles
            .filter(fileData => fileData.uploadData && fileData.uploadData.attachment_id)
            .map(fileData => ({
                attachment_id: fileData.uploadData.attachment_id,
                original_name: fileData.file.name,
                file_size: fileData.file.size,
                file_type: fileData.file.type
            }));
    }
    
    /**
     * Initialize the component (no mounting needed since HTML already exists)
     */
    mount(container) {
        // Component is already mounted since HTML exists in the page
        // Just return the component instance
        return this;
    }
    
    /**
     * Destroy the component
     */
    destroy() {
        if (this.element && this.element.parentNode) {
            this.element.parentNode.removeChild(this.element);
        }
    }
}

// Make it available globally
window.DentalFileUpload = DentalFileUpload;
